<?php
use yii\widgets\ActiveForm;
use yii\helpers\Html;
use yii\web\View;

/* @var $model app\models\Review */
/* @var $request app\models\Request */

// Регистрируем CSS стили
$css = <<< CSS
/* Основные цвета */
:root {
    --primary-black: #121212;
    --secondary-black: #1a1a1a;
    --dark-gray: #2d2d2d;
    --light-gray: #444;
    --primary-pink: #e83e8c;
    --secondary-pink: #ff6b9d;
    --soft-pink: rgba(232, 62, 140, 0.1);
    --text-primary: #f8f9fa;
    --text-secondary: #b0b0b0;
    --border-color: #333;
    --hover-black: #252525;
    --success-color: #28a745;
}

/* Контейнер формы отзыва */
.review-container {
    background-color: var(--primary-black);
    border-radius: 16px;
    padding: 30px;
    box-shadow: 0 15px 35px rgba(0, 0, 0, 0.5);
    border: 1px solid var(--border-color);
    margin-bottom: 30px;
    position: relative;
    overflow: hidden;
    max-width: 700px;
    margin: 0 auto;
}

.review-container::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 4px;
    background: linear-gradient(90deg, var(--primary-pink), var(--secondary-pink));
    z-index: 10;
}

.review-container::after {
    content: '';
    position: absolute;
    top: -50%;
    right: -50%;
    width: 200%;
    height: 200%;
    background: radial-gradient(circle, rgba(232, 62, 140, 0.05) 0%, rgba(232, 62, 140, 0) 70%);
    z-index: 0;
    pointer-events: none;
}

/* Заголовок формы */
.review-header {
    color: var(--text-primary);
    font-size: 28px;
    margin-bottom: 25px;
    padding-bottom: 20px;
    border-bottom: 1px solid var(--border-color);
    position: relative;
    font-weight: 600;
    z-index: 1;
}

.review-header::after {
    content: '';
    position: absolute;
    bottom: -1px;
    left: 0;
    width: 120px;
    height: 2px;
    background: linear-gradient(90deg, var(--primary-pink), transparent);
}

/* Информация об игре */
.game-review-info {
    background-color: var(--secondary-black);
    border-radius: 12px;
    padding: 20px;
    margin-bottom: 30px;
    border: 1px solid var(--border-color);
    position: relative;
    overflow: hidden;
    z-index: 1;
}

.game-review-info::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    width: 6px;
    height: 100%;
    background: linear-gradient(to bottom, var(--primary-pink), var(--secondary-pink));
}

.game-info-title {
    color: var(--text-primary);
    font-size: 18px;
    margin-bottom: 15px;
    font-weight: 600;
    display: flex;
    align-items: center;
    gap: 10px;
}

.game-info-title::before {
    content: '📝';
    font-size: 20px;
}

.game-details {
    display: flex;
    flex-wrap: wrap;
    gap: 15px;
    margin-top: 15px;
}

.game-detail-item {
    display: flex;
    flex-direction: column;
    flex: 1;
    min-width: 150px;
}

.game-detail-label {
    color: var(--text-secondary);
    font-size: 13px;
    margin-bottom: 5px;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.game-detail-value {
    color: var(--text-primary);
    font-size: 15px;
    font-weight: 500;
}

/* Анимация появления формы */
@keyframes fadeInReview {
    from {
        opacity: 0;
        transform: translateY(30px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.review-form-wrapper {
    animation: fadeInReview 0.8s ease-out forwards;
    position: relative;
    z-index: 1;
}

/* Поле для текста отзыва */
.review-text-group {
    margin-bottom: 30px;
}

.review-text-group label {
    color: var(--text-primary);
    font-weight: 500;
    margin-bottom: 10px;
    display: block;
    font-size: 16px;
    position: relative;
    padding-left: 25px;
}

.review-text-group label::before {
    content: '💬';
    position: absolute;
    left: 0;
    top: 0;
    font-size: 18px;
}

.review-textarea {
    width: 100%;
    min-height: 180px;
    background-color: var(--dark-gray);
    border: 1px solid var(--border-color);
    color: var(--text-primary);
    border-radius: 10px;
    padding: 15px;
    font-size: 15px;
    line-height: 1.6;
    resize: vertical;
    transition: all 0.3s ease;
    font-family: inherit;
}

.review-textarea:focus {
    border-color: var(--primary-pink);
    box-shadow: 0 0 0 3px rgba(232, 62, 140, 0.2);
    outline: none;
    background-color: var(--light-gray);
}

.review-textarea::placeholder {
    color: var(--text-secondary);
    opacity: 0.7;
}

/* Счетчик символов */
.char-counter {
    text-align: right;
    color: var(--text-secondary);
    font-size: 13px;
    margin-top: 5px;
    transition: color 0.3s ease;
}

.char-counter.warning {
    color: #ffc107;
}

.char-counter.error {
    color: #dc3545;
}

/* Рейтинг (если добавится в будущем) */
.rating-section {
    margin-bottom: 30px;
    padding: 20px;
    background-color: var(--secondary-black);
    border-radius: 10px;
    border: 1px solid var(--border-color);
}

.rating-title {
    color: var(--text-primary);
    font-weight: 500;
    margin-bottom: 15px;
    display: flex;
    align-items: center;
    gap: 10px;
}

.rating-title::before {
    content: '⭐';
    font-size: 18px;
}

.rating-stars {
    display: flex;
    gap: 5px;
    direction: rtl;
}

.rating-stars input {
    display: none;
}

.rating-stars label {
    font-size: 32px;
    color: var(--border-color);
    cursor: pointer;
    transition: all 0.2s ease;
    padding: 0 2px;
}

.rating-stars label:hover,
.rating-stars label:hover ~ label,
.rating-stars input:checked ~ label {
    color: #ffc107;
}

/* Кнопка отправки */
.btn-submit-review {
    background: linear-gradient(135deg, var(--success-color), #20c997);
    color: white;
    border: none;
    border-radius: 10px;
    padding: 16px 30px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 1px;
    font-size: 16px;
    transition: all 0.4s ease;
    box-shadow: 0 6px 20px rgba(40, 167, 69, 0.3);
    position: relative;
    overflow: hidden;
    display: block;
    width: 100%;
    cursor: pointer;
    margin-top: 20px;
}

.btn-submit-review::before {
    content: '';
    position: absolute;
    top: 0;
    left: -100%;
    width: 100%;
    height: 100%;
    background: linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.2), transparent);
    transition: left 0.7s ease;
}

.btn-submit-review:hover {
    transform: translateY(-3px);
    box-shadow: 0 12px 25px rgba(40, 167, 69, 0.4);
}

.btn-submit-review:hover::before {
    left: 100%;
}

.btn-submit-review:active {
    transform: translateY(-1px);
}

.btn-submit-review:disabled {
    opacity: 0.6;
    cursor: not-allowed;
    transform: none !important;
    box-shadow: none !important;
}

/* Иконка отправки */
.btn-submit-review::after {
    content: '📤';
    margin-left: 10px;
    font-size: 18px;
    vertical-align: middle;
}

/* Индикатор загрузки */
.review-loading {
    display: none;
    text-align: center;
    padding: 20px;
}

.review-loading.active {
    display: block;
}

.loading-spinner-review {
    width: 40px;
    height: 40px;
    border: 3px solid var(--border-color);
    border-top: 3px solid var(--success-color);
    border-radius: 50%;
    animation: spinReview 1s linear infinite;
    margin: 0 auto 15px;
}

@keyframes spinReview {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

/* Сообщение об успешной отправке */
.review-success {
    text-align: center;
    padding: 40px 20px;
    background-color: rgba(40, 167, 69, 0.1);
    border-radius: 10px;
    border: 1px solid rgba(40, 167, 69, 0.2);
    margin-top: 20px;
    display: none;
}

.review-success.active {
    display: block;
    animation: fadeInSuccess 0.5s ease-out;
}

@keyframes fadeInSuccess {
    from {
        opacity: 0;
        transform: translateY(-10px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.success-icon {
    font-size: 48px;
    color: var(--success-color);
    margin-bottom: 20px;
    display: block;
}

.success-text {
    color: var(--text-primary);
    font-size: 18px;
    font-weight: 500;
}

/* Адаптивность */
@media (max-width: 768px) {
    .review-container {
        padding: 20px;
        margin: 0 15px;
    }
    
    .review-header {
        font-size: 24px;
    }
    
    .game-details {
        flex-direction: column;
        gap: 10px;
    }
    
    .game-detail-item {
        min-width: 100%;
    }
    
    .review-textarea {
        min-height: 150px;
    }
    
    .btn-submit-review {
        padding: 14px 20px;
        font-size: 15px;
    }
}

@media (max-width: 480px) {
    .review-container {
        padding: 15px;
    }
    
    .review-header {
        font-size: 22px;
    }
    
    .game-review-info {
        padding: 15px;
    }
    
    .rating-stars label {
        font-size: 28px;
    }
}
CSS;

$this->registerCss($css);

// Регистрируем JavaScript для интерактивности
$js = <<< JS
$(document).ready(function() {
    // Счетчик символов для текстового поля
    var textarea = $('#review-text');
    var charCounter = $('.char-counter');
    var maxChars = 1000;
    
    if (textarea.length) {
        // Обновляем счетчик при загрузке
        updateCharCounter();
        
        // Обновляем счетчик при вводе
        textarea.on('input', function() {
            updateCharCounter();
        });
        
        function updateCharCounter() {
            var currentLength = textarea.val().length;
            var remaining = maxChars - currentLength;
            
            charCounter.text(currentLength + ' / ' + maxChars + ' символов');
            
            // Меняем цвет в зависимости от количества символов
            charCounter.removeClass('warning error');
            if (remaining < 100 && remaining >= 0) {
                charCounter.addClass('warning');
            } else if (remaining < 0) {
                charCounter.addClass('error');
            }
        }
    }
    
    // Анимация отправки формы
    $('form').on('submit', function(e) {
        var form = $(this);
        var textarea = form.find('textarea');
        var submitBtn = form.find('.btn-submit-review');
        
        // Проверка на минимальную длину отзыва
        if (textarea.val().trim().length < 10) {
            e.preventDefault();
            showError('Пожалуйста, напишите отзыв длиной не менее 10 символов.');
            return false;
        }
        
        // Проверка на максимальную длину
        if (textarea.val().length > 1000) {
            e.preventDefault();
            showError('Отзыв не должен превышать 1000 символов.');
            return false;
        }
        
        // Блокируем кнопку и показываем индикатор загрузки
        submitBtn.prop('disabled', true);
        submitBtn.html('<span class="loading-text">Отправка...</span>');
        
        // Имитация задержки отправки
        setTimeout(function() {
            // В реальном приложении здесь будет отправка формы
            // form.submit();
        }, 1500);
        
        return true;
    });
    
    // Функция для показа ошибки
    function showError(message) {
        // Создаем элемент ошибки
        var errorElement = $('<div class="review-error">').text(message);
        
        // Стили для ошибки
        errorElement.css({
            'background-color': 'rgba(220, 53, 69, 0.1)',
            'color': '#dc3545',
            'padding': '12px 15px',
            'border-radius': '8px',
            'border': '1px solid rgba(220, 53, 69, 0.2)',
            'margin-bottom': '20px',
            'font-weight': '500'
        });
        
        // Анимация появления
        errorElement.hide().insertAfter('.review-header').fadeIn(300);
        
        // Автоматическое скрытие через 5 секунд
        setTimeout(function() {
            errorElement.fadeOut(300, function() {
                $(this).remove();
            });
        }, 5000);
    }
    
    // Добавляем плавное появление элементов
    $('.review-form-wrapper > *').each(function(index) {
        $(this).css({
            'opacity': '0',
            'transform': 'translateY(20px)',
            'animation': 'fadeInReview 0.5s ease-out forwards',
            'animation-delay': (index * 0.1) + 's'
        });
    });
    
    // Добавляем звезды рейтинга (если нужно будет в будущем)
    var ratingHtml = `
        <div class="rating-section">
            <div class="rating-title">Оценка игры</div>
            <div class="rating-stars">
                <input type="radio" id="star5" name="rating" value="5">
                <label for="star5" title="Отлично">★</label>
                <input type="radio" id="star4" name="rating" value="4">
                <label for="star4" title="Хорошо">★</label>
                <input type="radio" id="star3" name="rating" value="3">
                <label for="star3" title="Нормально">★</label>
                <input type="radio" id="star2" name="rating" value="2">
                <label for="star2" title="Плохо">★</label>
                <input type="radio" id="star1" name="rating" value="1">
                <label for="star1" title="Ужасно">★</label>
            </div>
        </div>
    `;
    
    // Раскомментировать, если нужно добавить рейтинг:
    // $('.review-text-group').after(ratingHtml);
});
JS;

$this->registerJs($js, View::POS_READY);
?>

<div class="review-container">
    <h1 class="review-header">Оставить отзыв по игре: <?= Html::encode($request->game_name) ?></h1>

    <!-- Информация об игре -->
    <div class="game-review-info">
        <div class="game-info-title">Информация об игре</div>
        <div class="game-details">
            <div class="game-detail-item">
                <div class="game-detail-label">Вид игры</div>
                <div class="game-detail-value"><?= Html::encode($request->game_type) ?></div>
            </div>
            <div class="game-detail-item">
                <div class="game-detail-label">Дата</div>
                <div class="game-detail-value"><?= Html::encode($request->game_date) ?></div>
            </div>
            <div class="game-detail-item">
                <div class="game-detail-label">Время</div>
                <div class="game-detail-value"><?= Html::encode($request->game_time) ?></div>
            </div>
            <div class="game-detail-item">
                <div class="game-detail-label">Адрес</div>
                <div class="game-detail-value"><?= Html::encode($request->address) ?></div>
            </div>
        </div>
    </div>

    <div class="review-form-wrapper">
        <?php $form = ActiveForm::begin([
            'id' => 'review-form',
            'enableClientValidation' => true,
        ]); ?>

        <div class="review-text-group">
            <?= $form->field($model, 'text', [
                'template' => "{label}\n{input}\n<div class='char-counter'>0 / 1000 символов</div>\n{error}",
                'labelOptions' => ['class' => null]
            ])->textarea([
                'rows' => 6,
                'class' => 'review-textarea',
                'id' => 'review-text',
                'placeholder' => 'Поделитесь вашими впечатлениями от игры... Что понравилось, что можно улучшить?'
            ])->label('Ваш отзыв') ?>
        </div>

        <?= Html::submitButton('Сохранить отзыв', [
            'class' => 'btn-submit-review',
            'id' => 'submit-review-btn'
        ]) ?>

        <?php ActiveForm::end(); ?>
    </div>

    <!-- Индикатор загрузки -->
    <div class="review-loading">
        <div class="loading-spinner-review"></div>
        <p style="color: var(--text-secondary); margin-top: 10px;">Отправка отзыва...</p>
    </div>

    <!-- Сообщение об успешной отправке -->
    <div class="review-success">
        <span class="success-icon">✓</span>
        <p class="success-text">Ваш отзыв успешно отправлен!</p>
    </div>
</div>