<?php
use yii\helpers\Html;
use yii\web\View;

/* @var $requests app\models\Request[] */

$this->title = 'Мои заявки';

// Регистрируем CSS стили
$css = <<< CSS
/* Основные цвета */
:root {
    --primary-black: #121212;
    --secondary-black: #1a1a1a;
    --dark-gray: #2d2d2d;
    --light-gray: #444;
    --primary-pink: #e83e8c;
    --secondary-pink: #ff6b9d;
    --soft-pink: rgba(232, 62, 140, 0.1);
    --text-primary: #f8f9fa;
    --text-secondary: #b0b0b0;
    --border-color: #333;
    --hover-black: #252525;
    --warning-color: #ffc107;
    --success-color: #28a745;
    --info-color: #17a2b8;
}

/* Контейнер страницы */
.requests-container {
    background-color: var(--primary-black);
    border-radius: 16px;
    padding: 30px;
    box-shadow: 0 15px 35px rgba(0, 0, 0, 0.5);
    border: 1px solid var(--border-color);
    margin-bottom: 40px;
    position: relative;
    overflow: hidden;
}

.requests-container::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 4px;
    background: linear-gradient(90deg, var(--primary-pink), var(--secondary-pink));
    z-index: 10;
}

/* Заголовок страницы */
.page-header {
    color: var(--text-primary);
    font-size: 32px;
    margin-bottom: 30px;
    padding-bottom: 20px;
    border-bottom: 1px solid var(--border-color);
    position: relative;
    font-weight: 600;
    text-align: center;
}

.page-header::after {
    content: '';
    position: absolute;
    bottom: -1px;
    left: 50%;
    transform: translateX(-50%);
    width: 200px;
    height: 2px;
    background: linear-gradient(90deg, transparent, var(--primary-pink), transparent);
}

/* Анимация появления контента */
@keyframes fadeInContent {
    from {
        opacity: 0;
        transform: translateY(20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.content-wrapper {
    animation: fadeInContent 0.8s ease-out forwards;
}

/* Таблица заявок */
.requests-table {
    width: 100%;
    border-collapse: separate;
    border-spacing: 0;
    color: var(--text-primary);
    font-size: 15px;
    margin-bottom: 30px;
}

.requests-table th {
    background-color: var(--dark-gray);
    color: var(--text-primary);
    font-weight: 600;
    padding: 18px 15px;
    text-align: left;
    border: none;
    position: relative;
    text-transform: uppercase;
    font-size: 14px;
    letter-spacing: 0.5px;
    border-bottom: 2px solid var(--primary-pink);
}

.requests-table th:first-child {
    border-top-left-radius: 10px;
}

.requests-table th:last-child {
    border-top-right-radius: 10px;
}

.requests-table td {
    padding: 20px 15px;
    border-bottom: 1px solid var(--border-color);
    background-color: var(--secondary-black);
    transition: all 0.3s ease;
    position: relative;
}

/* Анимация строк таблицы */
.requests-table tbody tr {
    opacity: 0;
    animation: slideInRow 0.6s ease-out forwards;
}

@keyframes slideInRow {
    from {
        opacity: 0;
        transform: translateX(-20px);
    }
    to {
        opacity: 1;
        transform: translateX(0);
    }
}

.requests-table tbody tr:hover td {
    background-color: var(--hover-black);
    transform: translateY(-2px);
    box-shadow: 0 5px 15px rgba(0, 0, 0, 0.2);
}

.requests-table tbody tr:last-child td {
    border-bottom: none;
}

.requests-table tbody tr:last-child td:first-child {
    border-bottom-left-radius: 10px;
}

.requests-table tbody tr:last-child td:last-child {
    border-bottom-right-radius: 10px;
}

/* Стили для порядкового номера */
.request-number {
    color: var(--secondary-pink);
    font-weight: 600;
    font-size: 16px;
    text-align: center;
    width: 50px;
}

/* Стили для статусов */
.request-status {
    display: inline-block;
    padding: 6px 12px;
    border-radius: 20px;
    font-size: 13px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    min-width: 120px;
    text-align: center;
}

.status-pending {
    background-color: rgba(255, 193, 7, 0.2);
    color: var(--warning-color);
    border: 1px solid rgba(255, 193, 7, 0.3);
}

.status-confirmed {
    background-color: rgba(40, 167, 69, 0.2);
    color: var(--success-color);
    border: 1px solid rgba(40, 167, 69, 0.3);
}

.status-completed {
    background-color: rgba(23, 162, 184, 0.2);
    color: var(--info-color);
    border: 1px solid rgba(23, 162, 184, 0.3);
}

.status-cancelled {
    background-color: rgba(220, 53, 69, 0.2);
    color: #dc3545;
    border: 1px solid rgba(220, 53, 69, 0.3);
}

/* Кнопка отзыва */
.btn-review {
    background: linear-gradient(135deg, var(--warning-color), #ffd54f);
    color: #212529;
    border: none;
    border-radius: 8px;
    padding: 8px 16px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    font-size: 13px;
    transition: all 0.3s ease;
    box-shadow: 0 4px 12px rgba(255, 193, 7, 0.3);
    position: relative;
    overflow: hidden;
    display: inline-block;
    text-decoration: none;
    text-align: center;
}

.btn-review::after {
    content: '';
    position: absolute;
    top: 0;
    left: -100%;
    width: 100%;
    height: 100%;
    background: linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.3), transparent);
    transition: left 0.7s ease;
}

.btn-review:hover {
    color: #212529;
    text-decoration: none;
    transform: translateY(-2px);
    box-shadow: 0 6px 18px rgba(255, 193, 7, 0.4);
}

.btn-review:hover::after {
    left: 100%;
}

/* Сообщение об оставленном отзыве */
.review-done {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    color: var(--success-color);
    font-weight: 500;
    padding: 8px 12px;
    background-color: rgba(40, 167, 69, 0.1);
    border-radius: 8px;
    border: 1px solid rgba(40, 167, 69, 0.2);
}

.review-done::before {
    content: '✓';
    font-weight: bold;
    font-size: 16px;
}

/* Пустая таблица */
.empty-requests {
    text-align: center;
    padding: 60px 20px;
    background-color: var(--secondary-black);
    border-radius: 12px;
    margin-top: 20px;
    border: 2px dashed var(--border-color);
}

.empty-icon {
    font-size: 64px;
    color: var(--primary-pink);
    margin-bottom: 20px;
    display: block;
    opacity: 0.5;
}

.empty-title {
    color: var(--text-primary);
    font-size: 24px;
    margin-bottom: 15px;
    font-weight: 600;
}

.empty-text {
    color: var(--text-secondary);
    font-size: 16px;
    max-width: 400px;
    margin: 0 auto;
}

/* Индикаторы для важных полей */
.game-name {
    font-weight: 600;
    color: var(--text-primary);
    position: relative;
    padding-left: 20px;
}

.game-name::before {
    content: '🎮';
    position: absolute;
    left: 0;
    top: 50%;
    transform: translateY(-50%);
}

.game-price {
    font-weight: 600;
    color: var(--secondary-pink);
    font-size: 16px;
}

/* Адаптивность */
@media (max-width: 992px) {
    .requests-container {
        padding: 20px;
    }
    
    .page-header {
        font-size: 28px;
    }
    
    .requests-table {
        font-size: 14px;
    }
    
    .requests-table th,
    .requests-table td {
        padding: 15px 10px;
    }
    
    .request-status {
        min-width: 100px;
        font-size: 12px;
        padding: 5px 10px;
    }
}

@media (max-width: 768px) {
    .table-responsive {
        overflow-x: auto;
    }
    
    .requests-table {
        min-width: 700px;
    }
    
    .page-header {
        font-size: 24px;
    }
    
    .empty-icon {
        font-size: 48px;
    }
    
    .empty-title {
        font-size: 20px;
    }
}

/* Анимация загрузки для таблицы */
@keyframes pulse {
    0% { opacity: 0.6; }
    50% { opacity: 1; }
    100% { opacity: 0.6; }
}

.loading-row {
    animation: pulse 1.5s infinite;
}

/* Кастомный скроллбар */
::-webkit-scrollbar {
    width: 8px;
    height: 8px;
}

::-webkit-scrollbar-track {
    background: var(--dark-gray);
    border-radius: 4px;
}

::-webkit-scrollbar-thumb {
    background: var(--primary-pink);
    border-radius: 4px;
}

::-webkit-scrollbar-thumb:hover {
    background: var(--secondary-pink);
}
CSS;

$this->registerCss($css);

// Регистрируем JavaScript для анимаций
$js = <<< JS
$(document).ready(function() {
    // Анимация строк таблицы с задержкой
    $('.requests-table tbody tr').each(function(index) {
        $(this).css('animation-delay', (index * 0.1) + 's');
    });
    
    // Функция для определения стиля статуса
    function getStatusClass(status) {
        status = status.toLowerCase();
        if (status.includes('ожидание') || status.includes('pending')) {
            return 'status-pending';
        } else if (status.includes('подтвержден') || status.includes('confirmed') || status.includes('актив')) {
            return 'status-confirmed';
        } else if (status.includes('заверш') || status.includes('completed')) {
            return 'status-completed';
        } else if (status.includes('отмен') || status.includes('cancelled')) {
            return 'status-cancelled';
        }
        return 'status-pending';
    }
    
    // Применяем стили к статусам
    $('.request-status').each(function() {
        var statusText = $(this).text().trim();
        var statusClass = getStatusClass(statusText);
        $(this).addClass(statusClass);
    });
    
    // Добавляем иконки к типам игр
    $('td:nth-child(3)').each(function() {
        var gameType = $(this).text().trim().toLowerCase();
        var icon = '';
        
        switch(true) {
            case gameType.includes('футбол'):
                icon = '⚽';
                break;
            case gameType.includes('баскетбол'):
                icon = '🏀';
                break;
            case gameType.includes('волейбол'):
                icon = '🏐';
                break;
            case gameType.includes('теннис'):
                icon = '🎾';
                break;
            case gameType.includes('хоккей'):
                icon = '🏒';
                break;
            default:
                icon = '🎯';
        }
        
        $(this).html(icon + ' ' + $(this).text());
    });
    
    // Эффект при наведении на кнопку отзыва
    $('.btn-review').on('mouseenter', function() {
        $(this).addClass('animated');
    });
    
    // Показываем количество заявок в заголовке
    var requestCount = $('.requests-table tbody tr').length;
    if (requestCount > 0) {
        $('.page-header').append('<span class="badge-count">' + requestCount + '</span>');
        
        // Стили для бейджа с количеством
        $('<style>')
            .text('.badge-count { background: linear-gradient(135deg, var(--primary-pink), var(--secondary-pink)); color: white; font-size: 18px; padding: 4px 12px; border-radius: 20px; margin-left: 15px; vertical-align: middle; display: inline-block; animation: pulse 2s infinite; }')
            .appendTo('head');
    }
});
JS;

$this->registerJs($js, View::POS_READY);
?>

<div class="requests-container">
    <h1 class="page-header"><?= Html::encode($this->title) ?></h1>

    <div class="content-wrapper">
        <?php if (!empty($requests)): ?>
            <div class="table-responsive">
                <table class="requests-table">
                    <thead>
                    <tr>
                        <th>#</th>
                        <th>Игра</th>
                        <th>Вид</th>
                        <th>Дата</th>
                        <th>Время</th>
                        <th>Адрес</th>
                        <th>Цена</th>
                        <th>Статус</th>
                        <th>Действия</th>
                    </tr>
                    </thead>
                    <tbody>
                    <?php foreach ($requests as $i => $request): ?>
                        <tr>
                            <td class="request-number"><?= $i + 1 ?></td>
                            <td class="game-name"><?= Html::encode($request->game_name) ?></td>
                            <td><?= Html::encode($request->game_type) ?></td>
                            <td><?= Html::encode($request->game_date) ?></td>
                            <td><?= Html::encode($request->game_time) ?></td>
                            <td><?= Html::encode($request->address) ?></td>
                            <td class="game-price"><?= Html::encode($request->price) ?> ₽</td>
                            <td>
                                <span class="request-status"><?= $request->status ?></span>
                            </td>
                            <td>
                                <?php if (in_array($request->status, ['Завершена', 'Заказ завершен']) && !$request->review): ?>
                                    <?= Html::a('Оставить отзыв', ['site/review', 'request_id' => $request->id], [
                                        'class' => 'btn-review',
                                        'data' => ['request-id' => $request->id]
                                    ]) ?>
                                <?php elseif ($request->review): ?>
                                    <span class="review-done">Отзыв оставлен</span>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php else: ?>
            <div class="empty-requests">
                <span class="empty-icon">📝</span>
                <h3 class="empty-title">Заявок пока нет</h3>
                <p class="empty-text">У вас пока нет заявок на участие в играх. Запишитесь на игру из списка доступных!</p>
            </div>
        <?php endif; ?>
    </div>
</div>