<?php
use yii\helpers\Html;
use yii\web\View;

// Регистрируем CSS стили
$css = <<< CSS
/* Основные цвета */
:root {
    --primary-black: #121212;
    --secondary-black: #1a1a1a;
    --dark-gray: #2d2d2d;
    --light-gray: #444;
    --primary-pink: #e83e8c;
    --secondary-pink: #ff6b9d;
    --soft-pink: rgba(232, 62, 140, 0.1);
    --text-primary: #f8f9fa;
    --text-secondary: #b0b0b0;
    --border-color: #333;
    --hover-black: #252525;
    --success-color: #28a745;
    --warning-color: #ffc107;
    --danger-color: #dc3545;
    --info-color: #17a2b8;
}

/* Секции */
.section {
    background-color: var(--secondary-black);
    border-radius: 16px;
    padding: 40px;
    margin-bottom: 40px;
    border: 1px solid var(--border-color);
    box-shadow: 0 10px 30px rgba(0, 0, 0, 0.3);
    position: relative;
    overflow: hidden;
}

.section::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 4px;
    background: linear-gradient(90deg, var(--primary-pink), var(--secondary-pink));
    z-index: 1;
}

.section h2 {
    color: var(--text-primary);
    font-size: 32px;
    font-weight: 600;
    margin-bottom: 25px;
    position: relative;
    padding-bottom: 15px;
}

.section h2::after {
    content: '';
    position: absolute;
    bottom: 0;
    left: 0;
    width: 80px;
    height: 3px;
    background: linear-gradient(90deg, var(--primary-pink), transparent);
    border-radius: 2px;
}

/* Секция "О нас" */
#about p {
    color: var(--text-secondary);
    font-size: 18px;
    line-height: 1.7;
    margin-bottom: 30px;
}

.features-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 20px;
    margin-top: 30px;
}

.feature-card {
    background: var(--dark-gray);
    border-radius: 12px;
    padding: 25px;
    border: 1px solid var(--border-color);
    transition: all 0.3s ease;
}

.feature-card:hover {
    transform: translateY(-5px);
    border-color: var(--primary-pink);
    box-shadow: 0 10px 25px rgba(232, 62, 140, 0.15);
}

.feature-icon {
    font-size: 40px;
    margin-bottom: 15px;
    display: block;
}

.feature-title {
    color: var(--text-primary);
    font-size: 20px;
    font-weight: 600;
    margin-bottom: 10px;
}

.feature-text {
    color: var(--text-secondary);
    font-size: 15px;
    line-height: 1.6;
}

/* Каталог игр */
.categories-list {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 15px;
    margin-top: 20px;
}

.category-item {
    background: var(--dark-gray);
    border-radius: 10px;
    padding: 20px;
    border-left: 4px solid var(--primary-pink);
    transition: all 0.3s ease;
}

.category-item:hover {
    background: var(--hover-black);
    transform: translateX(5px);
}

.category-title {
    color: var(--text-primary);
    font-size: 18px;
    font-weight: 600;
    margin-bottom: 10px;
    display: flex;
    align-items: center;
    gap: 10px;
}

.category-title::before {
    content: '🎯';
    font-size: 20px;
}

.category-games {
    color: var(--text-secondary);
    font-size: 14px;
    line-height: 1.5;
}

.category-games li {
    margin-bottom: 5px;
    position: relative;
    padding-left: 15px;
}

.category-games li::before {
    content: '•';
    color: var(--primary-pink);
    position: absolute;
    left: 0;
}

/* Слайдер игр */
.games-slider {
    margin-top: 30px;
}

.games-row {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
    gap: 25px;
}

.game-slide {
    background: linear-gradient(135deg, var(--dark-gray), var(--light-gray));
    border-radius: 15px;
    padding: 30px;
    border: 1px solid var(--border-color);
    transition: all 0.4s ease;
    position: relative;
    overflow: hidden;
    text-align: center;
}

.game-slide::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: linear-gradient(45deg, transparent, rgba(232, 62, 140, 0.05), transparent);
    transform: translateX(-100%);
    transition: transform 0.6s ease;
}

.game-slide:hover {
    transform: translateY(-10px);
    box-shadow: 0 20px 40px rgba(232, 62, 140, 0.2);
    border-color: var(--primary-pink);
}

.game-slide:hover::before {
    transform: translateX(100%);
}

.game-icon {
    font-size: 60px;
    margin-bottom: 20px;
    display: block;
}

.game-name {
    color: var(--text-primary);
    font-size: 24px;
    font-weight: 600;
    margin-bottom: 15px;
}

.game-desc {
    color: var(--text-secondary);
    font-size: 15px;
    line-height: 1.6;
}

.game-rating {
    display: inline-flex;
    align-items: center;
    gap: 5px;
    background: rgba(232, 62, 140, 0.1);
    color: var(--primary-pink);
    padding: 8px 15px;
    border-radius: 20px;
    font-weight: 600;
    margin-top: 15px;
    font-size: 14px;
}

/* График игр - Таблица */
.table-container {
    background-color: var(--dark-gray);
    border-radius: 12px;
    overflow: hidden;
    border: 1px solid var(--border-color);
    margin-top: 20px;
}

.table-custom {
    width: 100%;
    border-collapse: separate;
    border-spacing: 0;
    color: var(--text-primary);
    font-size: 15px;
}

.table-custom th {
    background-color: var(--primary-black);
    color: var(--text-primary);
    font-weight: 600;
    padding: 18px 15px;
    text-align: left;
    border: none;
    position: relative;
    text-transform: uppercase;
    font-size: 14px;
    letter-spacing: 0.5px;
}

.table-custom th:not(:last-child)::after {
    content: '';
    position: absolute;
    right: 0;
    top: 50%;
    transform: translateY(-50%);
    height: 60%;
    width: 1px;
    background-color: var(--border-color);
}

.table-custom th:first-child {
    border-top-left-radius: 10px;
}

.table-custom th:last-child {
    border-top-right-radius: 10px;
}

.table-custom td {
    padding: 20px 15px;
    border-bottom: 1px solid var(--border-color);
    background-color: var(--secondary-black);
    transition: all 0.3s ease;
}

.table-custom tbody tr:hover td {
    background-color: var(--hover-black);
    transform: translateY(-2px);
    box-shadow: 0 5px 15px rgba(0, 0, 0, 0.2);
}

.table-custom tbody tr:last-child td {
    border-bottom: none;
}

.table-custom tbody tr:last-child td:first-child {
    border-bottom-left-radius: 10px;
}

.table-custom tbody tr:last-child td:last-child {
    border-bottom-right-radius: 10px;
}

.game-type {
    position: relative;
    padding-left: 25px;
    font-weight: 600;
}

.game-type::before {
    content: '';
    position: absolute;
    left: 10px;
    top: 50%;
    transform: translateY(-50%);
    width: 8px;
    height: 8px;
    background-color: var(--primary-pink);
    border-radius: 50%;
    box-shadow: 0 0 8px var(--primary-pink);
}

.game-date, .game-time {
    font-weight: 500;
}

.game-price {
    font-weight: 600;
    color: var(--secondary-pink);
    font-size: 16px;
}

/* Кнопка записи */
.btn-pink {
    background: linear-gradient(135deg, var(--primary-pink), var(--secondary-pink));
    color: white;
    border: none;
    border-radius: 8px;
    padding: 10px 20px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    font-size: 14px;
    transition: all 0.3s ease;
    box-shadow: 0 4px 12px rgba(232, 62, 140, 0.3);
    position: relative;
    overflow: hidden;
    display: inline-block;
    text-decoration: none;
    text-align: center;
}

.btn-pink::after {
    content: '';
    position: absolute;
    top: 0;
    left: -100%;
    width: 100%;
    height: 100%;
    background: linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.2), transparent);
    transition: left 0.7s ease;
}

.btn-pink:hover {
    color: white;
    text-decoration: none;
    transform: translateY(-3px);
    box-shadow: 0 8px 20px rgba(232, 62, 140, 0.4);
}

.btn-pink:hover::after {
    left: 100%;
}

.btn-pink:active {
    transform: translateY(-1px);
}

/* Профиль секция */
.profile-content {
    display: flex;
    align-items: center;
    justify-content: center;
    flex-direction: column;
    text-align: center;
    padding: 40px 20px;
    min-height: 200px;
}

.welcome-text {
    color: var(--text-primary);
    font-size: 28px;
    font-weight: 600;
    margin-bottom: 20px;
    background: linear-gradient(135deg, var(--primary-pink), var(--secondary-pink));
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

.user-name {
    color: var(--text-primary);
    font-size: 22px;
    font-weight: 500;
    margin-bottom: 30px;
}

.auth-buttons {
    display: flex;
    gap: 20px;
    flex-wrap: wrap;
    justify-content: center;
}

.btn-auth {
    padding: 14px 35px;
    border-radius: 10px;
    font-weight: 600;
    font-size: 16px;
    transition: all 0.3s ease;
    text-decoration: none;
    display: inline-flex;
    align-items: center;
    gap: 10px;
    border: 2px solid transparent;
}

.btn-login {
    background: linear-gradient(135deg, var(--primary-pink), var(--secondary-pink));
    color: white;
    box-shadow: 0 6px 20px rgba(232, 62, 140, 0.3);
}

.btn-login::before {
    content: '🔐';
    font-size: 18px;
}

.btn-register {
    background: transparent;
    color: var(--primary-pink);
    border-color: var(--primary-pink);
}

.btn-register::before {
    content: '📝';
    font-size: 18px;
}

.btn-auth:hover {
    transform: translateY(-3px);
    text-decoration: none;
}

.btn-login:hover {
    box-shadow: 0 12px 25px rgba(232, 62, 140, 0.4);
    color: white;
}

.btn-register:hover {
    background: var(--soft-pink);
    color: var(--primary-pink);
}

/* Подписка */
.subscribe-form {
    background: var(--dark-gray);
    border-radius: 12px;
    padding: 30px;
    border: 1px solid var(--border-color);
    max-width: 500px;
    margin: 0 auto;
}

.form-control-custom {
    width: 100%;
    background: var(--secondary-black);
    border: 1px solid var(--border-color);
    color: var(--text-primary);
    border-radius: 10px;
    padding: 15px 20px;
    font-size: 16px;
    transition: all 0.3s ease;
}

.form-control-custom:focus {
    border-color: var(--primary-pink);
    box-shadow: 0 0 0 3px rgba(232, 62, 140, 0.2);
    outline: none;
    background-color: var(--hover-black);
}

.form-control-custom::placeholder {
    color: var(--text-secondary);
}

.btn-subscribe {
    background: linear-gradient(135deg, var(--success-color), #20c997);
    color: white;
    border: none;
    border-radius: 10px;
    padding: 15px 30px;
    font-weight: 600;
    font-size: 16px;
    transition: all 0.3s ease;
    width: 100%;
    margin-top: 15px;
    box-shadow: 0 6px 20px rgba(40, 167, 69, 0.3);
    position: relative;
    overflow: hidden;
}

.btn-subscribe::before {
    content: '✉️';
    margin-right: 10px;
}

.btn-subscribe::after {
    content: '';
    position: absolute;
    top: 0;
    left: -100%;
    width: 100%;
    height: 100%;
    background: linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.2), transparent);
    transition: left 0.7s ease;
}

.btn-subscribe:hover {
    transform: translateY(-3px);
    box-shadow: 0 12px 25px rgba(40, 167, 69, 0.4);
}

.btn-subscribe:hover::after {
    left: 100%;
}

/* Модальное окно */
.modal-custom .modal-content {
    background: var(--secondary-black);
    border: 1px solid var(--border-color);
    border-radius: 16px;
    color: var(--text-primary);
}

.modal-custom .modal-header {
    border-bottom: 1px solid var(--border-color);
    padding: 25px 30px;
}

.modal-custom .modal-title {
    color: var(--text-primary);
    font-size: 24px;
    font-weight: 600;
}

.modal-custom .btn-close {
    filter: invert(1) grayscale(100%) brightness(200%);
}

.modal-custom .modal-body {
    padding: 40px 30px;
    text-align: center;
}

.modal-custom .modal-body .fs-5 {
    font-size: 20px !important;
    color: var(--text-secondary);
    line-height: 1.6;
}

.modal-custom .modal-footer {
    border-top: 1px solid var(--border-color);
    padding: 25px 30px;
    justify-content: center;
}

.btn-modal-close {
    background: transparent;
    color: var(--text-primary);
    border: 1px solid var(--border-color);
    padding: 12px 30px;
    border-radius: 8px;
    font-weight: 600;
    transition: all 0.3s ease;
}

.btn-modal-close:hover {
    background: var(--soft-pink);
    border-color: var(--primary-pink);
    color: var(--primary-pink);
}

/* Отзывы */
.reviews-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(350px, 1fr));
    gap: 25px;
    margin-top: 30px;
}

.review-card {
    background: var(--dark-gray);
    border-radius: 15px;
    padding: 30px;
    border: 1px solid var(--border-color);
    transition: all 0.3s ease;
    position: relative;
}

.review-card:hover {
    transform: translateY(-5px);
    border-color: var(--primary-pink);
    box-shadow: 0 15px 30px rgba(232, 62, 140, 0.1);
}

.review-card::before {
    content: '❝';
    position: absolute;
    top: 20px;
    left: 20px;
    color: var(--primary-pink);
    font-size: 40px;
    opacity: 0.3;
    font-family: serif;
}

.review-text {
    color: var(--text-primary);
    font-size: 16px;
    line-height: 1.7;
    margin-bottom: 20px;
    position: relative;
    z-index: 1;
}

.review-meta {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding-top: 20px;
    border-top: 1px solid var(--border-color);
}

.review-date {
    color: var(--text-secondary);
    font-size: 14px;
}

.review-rating {
    color: #ffc107;
    font-size: 16px;
}

/* Контакты */
.contacts-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 25px;
    margin-top: 30px;
}

.contact-item {
    background: var(--dark-gray);
    border-radius: 12px;
    padding: 25px;
    border-left: 4px solid var(--primary-pink);
    transition: all 0.3s ease;
}

.contact-item:hover {
    transform: translateY(-5px);
    background: var(--hover-black);
}

.contact-icon {
    font-size: 32px;
    margin-bottom: 15px;
    display: block;
}

.contact-label {
    color: var(--text-secondary);
    font-size: 14px;
    text-transform: uppercase;
    letter-spacing: 1px;
    margin-bottom: 5px;
}

.contact-value {
    color: var(--text-primary);
    font-size: 18px;
    font-weight: 500;
}

/* Анимации */
@keyframes fadeInUp {
    from {
        opacity: 0;
        transform: translateY(30px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.section {
    animation: fadeInUp 0.6s ease-out forwards;
    opacity: 0;
}

#about { animation-delay: 0.1s; }
#catalog { animation-delay: 0.2s; }
#slider { animation-delay: 0.3s; }
#schedule { animation-delay: 0.4s; }
#profile { animation-delay: 0.5s; }
#subscribe { animation-delay: 0.6s; }
#reviews { animation-delay: 0.7s; }
#contacts { animation-delay: 0.8s; }

/* Адаптивность */
@media (max-width: 768px) {
    .section {
        padding: 30px 20px;
    }
    
    .section h2 {
        font-size: 28px;
    }
    
    .games-row {
        grid-template-columns: 1fr;
    }
    
    .reviews-grid {
        grid-template-columns: 1fr;
    }
    
    .contacts-grid {
        grid-template-columns: 1fr;
    }
    
    .table-custom {
        font-size: 14px;
    }
    
    .table-custom th,
    .table-custom td {
        padding: 15px 10px;
    }
    
    .auth-buttons {
        flex-direction: column;
        width: 100%;
        max-width: 300px;
    }
    
    .btn-auth {
        width: 100%;
        justify-content: center;
    }
}

@media (max-width: 576px) {
    .section {
        padding: 25px 15px;
    }
    
    .section h2 {
        font-size: 24px;
    }
    
    .features-grid {
        grid-template-columns: 1fr;
    }
    
    .categories-list {
        grid-template-columns: 1fr;
    }
}
CSS;

$this->registerCss($css);

// Регистрируем JavaScript
$js = <<< JS
$(document).ready(function() {
    // Открытие модального окна подписки
    window.openSubscribeModal = function() {
        const emailInput = $('#subscribe-email');
        const email = emailInput.val().trim();
        
        if (!email) {
            showNotification('Пожалуйста, введите email адрес', 'warning');
            emailInput.focus();
            return;
        }
        
        if (!validateEmail(email)) {
            showNotification('Пожалуйста, введите корректный email', 'warning');
            emailInput.focus();
            return;
        }
        
        // Показываем модальное окно
        const modal = new bootstrap.Modal(document.getElementById('subscribeModal'));
        modal.show();
        
        // Очищаем поле ввода
        emailInput.val('');
        
        // В реальном приложении здесь будет отправка данных на сервер
        console.log('Email подписки:', email);
    };
    
    // Валидация email
    function validateEmail(email) {
        const re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return re.test(email);
    }
    
    // Показать уведомление
    function showNotification(message, type) {
        const notification = $('<div class="notification">')
            .text(message)
            .css({
                'position': 'fixed',
                'top': '20px',
                'right': '20px',
                'background': type === 'warning' ? 
                    'linear-gradient(135deg, var(--warning-color), #ffce3a)' : 
                    'linear-gradient(135deg, var(--success-color), #20c997)',
                'color': 'white',
                'padding': '15px 25px',
                'borderRadius': '8px',
                'boxShadow': '0 5px 15px rgba(0,0,0,0.3)',
                'zIndex': '9999',
                'transform': 'translateX(150%)',
                'transition': 'transform 0.3s ease',
                'maxWidth': '400px',
                'fontWeight': '500'
            })
            .appendTo('body');
        
        notification.css('transform', 'translateX(0)');
        
        setTimeout(function() {
            notification.css('transform', 'translateX(150%)');
            setTimeout(function() {
                notification.remove();
            }, 300);
        }, 3000);
    }
    
    // Анимация появления элементов при скролле
    function animateOnScroll() {
        $('.section').each(function() {
            const elementTop = $(this).offset().top;
            const elementBottom = elementTop + $(this).outerHeight();
            const viewportTop = $(window).scrollTop();
            const viewportBottom = viewportTop + $(window).height();
            
            if (elementBottom > viewportTop && elementTop < viewportBottom) {
                $(this).addClass('animated');
            }
        });
    }
    
    $(window).on('scroll', animateOnScroll);
    animateOnScroll(); // Запускаем при загрузке
    
    // Обработка нажатия Enter в поле email
    $('#subscribe-email').on('keypress', function(e) {
        if (e.which === 13) {
            e.preventDefault();
            openSubscribeModal();
        }
    });
    
    // Добавляем эффект для карточек отзывов
    $('.review-card').hover(
        function() {
            $(this).find('.review-text').css('color', 'var(--text-primary)');
        },
        function() {
            $(this).find('.review-text').css('color', 'var(--text-primary)');
        }
    );
    
    // Стиль для уведомлений
    $('<style>')
        .text(`
            .section.animated {
                opacity: 1 !important;
            }
        `)
        .appendTo('head');
});
JS;

$this->registerJs($js, View::POS_READY);
?>

<!-- Секция "О нас" -->
<section id="about" class="section">
    <h2>О нас</h2>
    <p>
        Мы организуем настольные игры для компаний любого уровня —
        от новичков до профессионалов. Уютная атмосфера, лучшие игры,
        живое общение.
    </p>

    <div class="features-grid">
        <div class="feature-card">
            <span class="feature-icon">🎯</span>
            <h3 class="feature-title">Для всех уровней</h3>
            <p class="feature-text">Подходим как новичкам, так и опытным игрокам. Объясним правила и поможем освоиться.</p>
        </div>

        <div class="feature-card">
            <span class="feature-icon">🎲</span>
            <h3 class="feature-title">Большая коллекция</h3>
            <p class="feature-text">Более 100 настольных игр разных жанров и сложностей. Всегда есть из чего выбрать.</p>
        </div>

        <div class="feature-card">
            <span class="feature-icon">👥</span>
            <h3 class="feature-title">Живое общение</h3>
            <p class="feature-text">Уютная атмосфера для знакомств и общения. Находите новых друзей и единомышленников.</p>
        </div>

        <div class="feature-card">
            <span class="feature-icon">☕</span>
            <h3 class="feature-title">Комфортная зона</h3>
            <p class="feature-text">Удобные столы, мягкие кресла, напитки и закуски. Всё для приятного времяпрепровождения.</p>
        </div>
    </div>
</section>

<!-- Каталог настольных игр -->
<section id="catalog" class="section">
    <h2>Каталог настольных игр</h2>

    <div class="categories-list">
        <div class="category-item">
            <h3 class="category-title">Классические</h3>
            <ul class="category-games">
                <li>Шахматы</li>
                <li>Шашки</li>
                <li>Нарды</li>
                <li>Домино</li>
            </ul>
        </div>

        <div class="category-item">
            <h3 class="category-title">Карточные</h3>
            <ul class="category-games">
                <li>Шакал</li>
                <li>Взрывные котята</li>
                <li>Свинтус</li>
                <li>Уно</li>
            </ul>
        </div>

        <div class="category-item">
            <h3 class="category-title">Экономические</h3>
            <ul class="category-games">
                <li>Монополия</li>
                <li>Маракайбо</li>
                <li>Каркассон</li>
                <li>Колонизаторы</li>
            </ul>
        </div>

        <div class="category-item">
            <h3 class="category-title">Стратегии</h3>
            <ul class="category-games">
                <li>Каркассон</li>
                <li>Улей</li>
                <li>Рокуган</li>
                <li>Манчкин</li>
            </ul>
        </div>

        <div class="category-item">
            <h3 class="category-title">Для детей</h3>
            <ul class="category-games">
                <li>Кубопрыги</li>
                <li>Фотовечеринка</li>
                <li>Доббль</li>
                <li>Тик Так Бумм</li>
            </ul>
        </div>
    </div>
</section>

<!-- Слайдер популярных игр -->
<section id="slider" class="section">
    <h2>Популярные игры</h2>

    <div class="games-slider">
        <div class="games-row">
            <div class="game-slide">
                <span class="game-icon">🎲</span>
                <h3 class="game-name">Каркассон</h3>
                <p class="game-desc">Стратегическая игра о строительстве средневековых городов и дорог. Простые правила, глубокая стратегия.</p>
                <div class="game-rating">
                    <span>★★★★★</span>
                    <span>4.8/5</span>
                </div>
            </div>

            <div class="game-slide">
                <span class="game-icon">🃏</span>
                <h3 class="game-name">Взрывные котята</h3>
                <p class="game-desc">Весёлая карточная игра в русской рулетке. Избегайте взрывных котят и используйте карты действий.</p>
                <div class="game-rating">
                    <span>★★★★★</span>
                    <span>4.7/5</span>
                </div>
            </div>

            <div class="game-slide">
                <span class="game-icon">♟</span>
                <h3 class="game-name">Шахматы</h3>
                <p class="game-desc">Классическая игра для развития стратегического мышления. Турниры для всех уровней подготовки.</p>
                <div class="game-rating">
                    <span>★★★★★</span>
                    <span>5.0/5</span>
                </div>
            </div>
        </div>
    </div>
</section>

<!-- График ближайших игр -->
<section id="schedule" class="section">
    <h2>Ближайшие игры</h2>

    <div class="table-container">
        <table class="table-custom">
            <thead>
            <tr>
                <th>Вид</th>
                <th>Игра</th>
                <th>Дата</th>
                <th>Время</th>
                <th>Адрес</th>
                <th>Цена</th>
                <th></th>
            </tr>
            </thead>
            <tbody>
            <?php if (!empty($games)): ?>
                <?php foreach ($games as $game): ?>
                    <tr>
                        <td class="game-type"><?= Html::encode($game->game_type) ?></td>
                        <td><?= Html::encode($game->game_name) ?></td>
                        <td class="game-date"><?= Html::encode($game->game_date) ?></td>
                        <td class="game-time"><?= Html::encode($game->game_time) ?></td>
                        <td><?= Html::encode($game->address) ?></td>
                        <td class="game-price"><?= Html::encode($game->price) ?> ₽</td>
                        <td>
                            <?= Html::a(
                                'Записаться',
                                ['site/create-request', 'id' => $game->id],
                                ['class' => 'btn-pink']
                            ) ?>
                        </td>
                    </tr>
                <?php endforeach; ?>
            <?php else: ?>
                <tr>
                    <td colspan="7" style="text-align: center; color: var(--text-secondary); padding: 40px;">
                        На данный момент нет запланированных игр
                    </td>
                </tr>
            <?php endif; ?>
            </tbody>
        </table>
    </div>
</section>

<!-- Профиль -->
<section id="profile" class="section">
    <h2>Профиль</h2>

    <div class="profile-content">
        <?php if (Yii::$app->user->isGuest): ?>
            <h3 class="welcome-text">Добро пожаловать!</h3>
            <p style="color: var(--text-secondary); margin-bottom: 30px; font-size: 16px;">
                Войдите или зарегистрируйтесь, чтобы записываться на игры, <br>
                просматривать свои заявки и получать уведомления.
            </p>

            <div class="auth-buttons">
                <a href="../site/login" class="btn-auth btn-login">Войти в аккаунт</a>
                <a href="../site/register" class="btn-auth btn-register">Регистрация</a>
            </div>
        <?php else: ?>
            <h3 class="welcome-text">С возвращением!</h3>
            <p class="user-name">Здравствуйте, <?= Html::encode(Yii::$app->user->identity->fio) ?></p>

            <div style="display: flex; gap: 20px; flex-wrap: wrap; justify-content: center;">
                <a href="../site/requests" class="btn-pink">Мои заявки</a>
                <a href="../site/games" class="btn-auth btn-register">Все игры</a>
            </div>
        <?php endif; ?>
    </div>
</section>

<!-- Подписка -->
<section id="subscribe" class="section">
    <h2>Подписаться на новости</h2>

    <div class="subscribe-form">
        <input type="email"
               id="subscribe-email"
               class="form-control-custom"
               placeholder="Введите ваш e-mail"
               required>

        <button type="button"
                class="btn-subscribe"
                onclick="openSubscribeModal()">
            Подписаться на новости
        </button>
    </div>
</section>

<!-- Модальное окно подписки -->
<div class="modal fade modal-custom" id="subscribeModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Спасибо за подписку! ❤️</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>

            <div class="modal-body">
                <p class="fs-5">
                    Мы отправили письмо с подтверждением на вашу электронную почту.<br><br>
                    Теперь вы будете первыми узнавать о новых играх, специальных акциях и предстоящих мероприятиях нашего клуба!
                </p>
            </div>

            <div class="modal-footer">
                <button type="button" class="btn-modal-close" data-bs-dismiss="modal">
                    Закрыть
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Отзывы участников -->
<section id="reviews" class="section">
    <h2>Отзывы участников</h2>

    <div class="reviews-grid">
        <?php if (!empty($reviews)): ?>
            <?php foreach ($reviews as $review): ?>
                <div class="review-card">
                    <p class="review-text"><?= nl2br(Html::encode($review->text)) ?></p>
                    <div class="review-meta">
                        <div class="review-date">
                            <?= date('d.m.Y', strtotime($review->created_at)) ?>
                        </div>
                        <div class="review-rating">
                            ★★★★★
                        </div>
                    </div>
                </div>
            <?php endforeach; ?>
        <?php else: ?>
            <div style="grid-column: 1/-1; text-align: center; padding: 40px; color: var(--text-secondary);">
                <p style="font-size: 18px; margin-bottom: 20px;">Пока нет отзывов</p>
                <p>Будьте первым, кто оставит отзыв после посещения нашего клуба!</p>
            </div>
        <?php endif; ?>
    </div>
</section>

<!-- Контакты -->
<section id="contacts" class="section">
    <h2>Контакты</h2>

    <div class="contacts-grid">
        <div class="contact-item">
            <span class="contact-icon">📍</span>
            <div class="contact-label">Адрес</div>
            <div class="contact-value">г. Москва, ул. Ленина, 10</div>
        </div>

        <div class="contact-item">
            <span class="contact-icon">📞</span>
            <div class="contact-label">Телефон</div>
            <div class="contact-value">+7 (999) 123-45-67</div>
        </div>

        <div class="contact-item">
            <span class="contact-icon">✉️</span>
            <div class="contact-label">Email</div>
            <div class="contact-value">info@nastolka.ru</div>
        </div>

        <div class="contact-item">
            <span class="contact-icon">⏰</span>
            <div class="contact-label">Режим работы</div>
            <div class="contact-value">Ежедневно 12:00-23:00</div>
        </div>
    </div>
</section>